#include <SPIFFS.h>
#include <Arduino.h>
#include <SPI.h>
#include "isp.h"

// Определение макросов и констант
#define USE_HEAP // Использование динамической памяти (кучи)

#define FPSTR(s)  ((__FlashStringHelper*)(s)) // Макрос для работы со строками в PROGMEM

// Типы записей в HEX-файле
enum hextype_t : uint8_t { HEX_BIN = 0, HEX_END, HEX_SEGMENT, HEX_START, HEX_EXTADDR, HEX_START32 };

// Статусы парсинга HEX-файла
enum hexparse_t : uint8_t { HEX_OK,
#ifdef USE_HEAP
  HEX_NOMEM, // Ошибка выделения памяти
#endif
  HEX_TOOSHORT, HEX_WRONGSTART, HEX_WRONGLEN, HEX_WRONGADDRHI, HEX_WRONGADDRLO, HEX_WRONGTYPE, HEX_WRONGDATA, HEX_WRONGCRC };

// Пины для кнопок и светодиодов
constexpr uint8_t BTN1_PIN = 0;  // Кнопка 1  
constexpr uint8_t BTN2_PIN = 15;  // Кнопка 2  

constexpr bool LED_LEVEL = LOW;  // Уровень сигнала для включения светодиода

constexpr uint8_t HEX_PAGE_SIZE = 16; // Размер страницы для HEX-файла
constexpr uint32_t BLINK_TIME = 50;   // Время мигания светодиода (в мс)

// Имена файлов в PROGMEM
static const char FUSES_NAME[] PROGMEM = "/fuses.txt";
static const char FUSES_BACKUP_NAME[] PROGMEM = "/fuses.txt";
static const char EEPROM_NAME[] PROGMEM = "/eeprom.hex";
static const char EEPROM_BACKUP_NAME[] PROGMEM = "/eeprom.hex";
static const char FIRMWARE_NAME[] PROGMEM = "/firmware.hex";
static const char FIRMWARE_BACKUP_NAME[] PROGMEM = "/firmware.hex";

// Сообщения об ошибках и успехе
static const char FAIL_OR_OK[][6] PROGMEM = { "FAIL!", "Done" };
static const char HEX_LINE_HAS[] PROGMEM = "\r\nHEX line has ";

File f; // Файл для работы с SPIFFS
bool error = false; // Флаг ошибки

// Функция проверки существования файла
static bool fexists(PGM_P fileName) {
  char name[13];
  strcpy_P(name, fileName); // Копируем имя файла из PROGMEM
  return SPIFFS.exists(name);   // Проверяем существование файла
}

// Функция чтения строки из файла до терминатора
static uint8_t freadUntil(char *str, uint8_t size, char terminator, char tail) {
  size = f.readBytesUntil(terminator, str, size - 1); // Чтение до терминатора
  if (size && (str[size - 1] == tail)) // Удаление хвостового символа
    --size;
  str[size] = '\0'; // Завершаем строку
  return size;
}

// Функция парсинга HEX-числа
static bool parseHexNum(const char *str, uint8_t &num) {
  uint8_t result;
  if ((*str >= '0') && (*str <= '9'))
    result = *str - '0';
  else if ((*str >= 'A') && (*str <= 'F'))
    result = *str - 'A' + 10;
  else if ((*str >= 'a') && (*str <= 'f'))
    result = *str - 'a' + 10;
  else
    return false;
  result <<= 4;
  ++str;
  if ((*str >= '0') && (*str <= '9'))
    result |= *str - '0';
  else if ((*str >= 'A') && (*str <= 'F'))
    result |= *str - 'A' + 10;
  else if ((*str >= 'a') && (*str <= 'f'))
    result |= *str - 'a' + 10;
  else
    return false;
  num = result;
  return true;
}

// Функция парсинга строки HEX-файла
static hexparse_t parseHexLine(uint8_t &len, uint16_t &addr, hextype_t &type, uint8_t *data) {
  constexpr uint8_t LINE_SIZE = HEX_PAGE_SIZE * 2 + 13 + 1;

#ifdef USE_HEAP
  char *str = new char[LINE_SIZE]; // Выделяем память в куче
#else
  char str[LINE_SIZE]; // Используем статический массив
#endif
  uint8_t l, crc;
  hexparse_t result;

  if (str) {
    l = freadUntil(str, LINE_SIZE, '\n', '\r'); // Читаем строку
    if (l < 11) {
      result = HEX_TOOSHORT; // Слишком короткая строка
      goto free_and_return;
    }
    if (str[0] != ':') {
      result = HEX_WRONGSTART; // Неправильный стартовый символ
      goto free_and_return;
    }
    if ((!parseHexNum(&str[1], len)) || (len > HEX_PAGE_SIZE)) {
      result = HEX_WRONGLEN; // Неправильная длина
      goto free_and_return;
    }
    crc = len;
    if (!parseHexNum(&str[3], l)) {
      result = HEX_WRONGADDRHI; // Неправильный старший байт адреса
      goto free_and_return;
    }
    addr = l << 8;
    crc += l;
    if (!parseHexNum(&str[5], l)) {
      result = HEX_WRONGADDRLO; // Неправильный младший байт адреса
      goto free_and_return;
    }
    addr |= l;
    crc += l;
    if ((!parseHexNum(&str[7], l)) || (l > HEX_START32)) {
      result = HEX_WRONGTYPE; // Неправильный тип записи
      goto free_and_return;
    }
    type = (hextype_t)l;
    crc += l;
    for (uint8_t i = 0; i < len; ++i) {
      if (!parseHexNum(&str[9 + i * 2], data[i])) {
        result = HEX_WRONGDATA; // Неправильные данные
        goto free_and_return;
      }
      crc += data[i];
    }
    crc = 0 - crc;
    if ((!parseHexNum(&str[9 + len * 2], l)) || (l != crc)) {
      result = HEX_WRONGCRC; // Неправильная контрольная сумма
      goto free_and_return;
    }
    result = HEX_OK; // Успешный парсинг
  free_and_return:
#ifdef USE_HEAP
    delete[] str; // Освобождаем память
#endif
  }
  return result;
}

// Функция вывода ошибки парсинга
static void printParseError(hexparse_t parse) {
  switch (parse) {
#ifdef USE_HEAP
    case HEX_NOMEM:
      Serial.println(F("\r\nNot enough memory!"));
      break;
#endif
    case HEX_TOOSHORT:
      Serial.println(F("\r\nHEX line too short!"));
      break;
    case HEX_WRONGSTART:
      Serial.print(FPSTR(HEX_LINE_HAS));
      Serial.println(F("wrong start!"));
      break;
    case HEX_WRONGLEN:
      Serial.print(FPSTR(HEX_LINE_HAS));
      Serial.println(F("wrong length!"));
      break;
    case HEX_WRONGADDRHI:
    case HEX_WRONGADDRLO:
      Serial.print(FPSTR(HEX_LINE_HAS));
      Serial.println(F("wrong address!"));
      break;
    case HEX_WRONGTYPE:
      Serial.print(FPSTR(HEX_LINE_HAS));
      Serial.println(F("wrong type!"));
      break;
    case HEX_WRONGDATA:
      Serial.print(FPSTR(HEX_LINE_HAS));
      Serial.println(F("wrong data!"));
      break;
    case HEX_WRONGCRC:
      Serial.print(FPSTR(HEX_LINE_HAS));
      Serial.println(F("wrong CRC!"));
      break;
    default:
      break;
  }
}

// Функция преобразования числа в HEX-строку
static const char *hex(char *str, uint8_t value) {
  uint8_t d;
  d = value / 16;
  if (d > 9)
    str[0] = 'A' + d - 10;
  else
    str[0] = '0' + d;
  d = value & 0x0F;
  if (d > 9)
    str[1] = 'A' + d - 10;
  else
    str[1] = '0' + d;
  str[2] = '\0';
  return str;
}

// Функция вычисления длины данных
static uint8_t dataLength(const uint8_t *data, uint8_t size) {
  while (size) {
    if (data[size - 1] != 0xFF)
      break;
    --size;
  }
  return size;
}

// Функция вывода процентов выполнения
static void printPercent(uint8_t percent) {
  if (percent < 100)
    Serial.write(' ');
  if (percent < 10)
    Serial.write(' ');
  Serial.print(percent);

// Преобразуем переменную percent в строку
String percentString = String(percent) + "%";

// Отображаем строку на дисплее
//LD.printString_6x8(percentString.c_str(), 0, 4);
delay(100);
}

// Функция записи fuse-ов в файл
static bool dumpFuses(PGM_P fileName) {
  char name[13];
  strcpy_P(name, fileName);
  f = SPIFFS.open(name, FILE_WRITE);
  if (f) {
    f.print(F("LB:"));
    f.println(hex(name, ispReadLockBits()));
    f.print(F("L:"));
    f.print(hex(name, ispReadLowFuseBits()));
    f.print(F(";H:"));
    f.print(hex(name, ispReadHighFuseBits()));
    f.print(F(";E:"));
    f.println(hex(name, ispReadExtFuseBits()));
    f.close();
    return true;
  }
  return false;
}

// Функция программирования fuse-ов из файла
static bool programFuses(PGM_P fileName, bool lock = false) {
  constexpr uint8_t STR_SIZE = 17;
  char name[13];
  bool result = false;

  strcpy_P(name, fileName);
  f = SPIFFS.open(name, FILE_READ);
  if (f) {
#ifdef USE_HEAP
    char *str = new char[STR_SIZE];
#else
    char str[STR_SIZE];
#endif
    uint8_t lb, lf, hf, ef;

    if (str) {
      if ((freadUntil(str, STR_SIZE, '\n', '\r') == 5) &&
        (!strncmp_P(str, PSTR("LB:"), 3)) && parseHexNum(&str[3], lb)) {
        if ((freadUntil(str, STR_SIZE, '\n', '\r') == 14) &&
          (!strncmp_P(str, PSTR("L:"), 2)) && parseHexNum(&str[2], lf) &&
          (!strncmp_P(&str[5], PSTR("H:"), 2)) && parseHexNum(&str[7], hf) &&
          (!strncmp_P(&str[10], PSTR("E:"), 2)) && parseHexNum(&str[12], ef)) {
          ispWriteLowFuseBits(lf);
          ispReset();
          result = ispBegin();
          if (result) {
            ispWriteHighFuseBits(hf);
            ispReset();
            result = ispBegin();
          }
          if (result) {
            ispWriteExtFuseBits(ef);
            ispReset();
            result = ispBegin();
          }
          if (result && lock) {
            ispWriteLockBits(lb);
            ispReset();
            result = ispBegin();
          }
        }
      }
#ifdef USE_HEAP
      delete[] str;
#endif
    }
    f.close();
  }
  return result;
}

// Функция записи EEPROM в файл
static bool dumpEeprom(PGM_P fileName) {
  char name[13];
  bool result = false;

  strcpy_P(name, fileName);
  f = SPIFFS.open(name, FILE_WRITE);
  if (f) {
#ifdef USE_HEAP
    uint8_t *data = new uint8_t[HEX_PAGE_SIZE];
#else
    uint8_t data[HEX_PAGE_SIZE];
#endif

    if (data) {
      for (uint16_t addr = 0; addr < EEPROM_SIZE; addr += HEX_PAGE_SIZE) {
        uint8_t len;

        for (uint8_t i = 0; i < HEX_PAGE_SIZE; ++i) {
          data[i] = ispReadEeprom(addr + i);
        }
        len = dataLength(data, HEX_PAGE_SIZE);
        if (len) {
          uint8_t crc;

          crc = len + (addr / 256) + (addr & 0xFF);
          for (uint8_t i = 0; i < len; ++i) {
            crc += data[i];
          }
          f.print(':');
          f.print(hex(name, len));
          f.print(hex(name, addr / 256));
          f.print(hex(name, addr));
          f.print(F("00"));
          for (uint8_t i = 0; i < len; ++i) {
            f.print(hex(name, data[i]));
          }
          f.println(hex(name, 0 - crc));
        }
        if (!(addr % 128))
          printPercent((uint32_t)addr * 100 / EEPROM_SIZE);
      }
#ifdef USE_HEAP
      delete[] data;
#endif
      f.println(F(":00000001FF"));
      result = true;
      Serial.println(FPSTR(FAIL_OR_OK[1]));
    }
    f.close();
  }
  return result;
}

// Функция программирования EEPROM из файла
static bool programEeprom(PGM_P fileName) {
  char name[13];
  bool result = false;

  strcpy_P(name, fileName);
  f = SPIFFS.open(name, FILE_READ);
  if (f) {
#ifdef USE_HEAP
    uint8_t *data = new uint8_t[HEX_PAGE_SIZE];
#else
    uint8_t data[HEX_PAGE_SIZE];
#endif
    uint16_t addr;
    hexparse_t parse;
    hextype_t type;
    uint8_t len;
    bool ok;

    if (data) {
      do {
        parse = parseHexLine(len, addr, type, data);
        if ((ok = (parse == HEX_OK))) {
          if (type == HEX_BIN) {
            if (addr + len <= EEPROM_SIZE) {
              for (uint8_t i = 0; i < len; ++i) {
                if (!ispWriteEeprom(addr + i, data[i], true)) {
                  Serial.println(F("\r\nEEPROM write error!"));
                  ok = false;
                  break;
                }
              }
            } else {
              Serial.print(FPSTR(HEX_LINE_HAS));
              Serial.println(F("wrong EEPROM address!"));
              ok = false;
            }
          } else if (type == HEX_END) {
            if ((len == 0) && (addr == 0))
              break;
            else {
              Serial.print(FPSTR(HEX_LINE_HAS));
              Serial.println(F("wrong END!"));
              ok = false;
            }
          } else {
            Serial.print(FPSTR(HEX_LINE_HAS));
            Serial.println(F("unexpected type!"));
            ok = false;
          }

        } else {
          printParseError(parse);
        }
      } while (ok);
#ifdef USE_HEAP
      delete[] data;
#endif

      result = ok;
    }
    f.close();
  }
  return result;
}
 
static bool dumpFlash(PGM_P fileName) {
  char name[13];
  bool result = false;

  strcpy_P(name, fileName);
  f = SPIFFS.open(name, FILE_WRITE);
  if (f) {
#ifdef USE_HEAP
    uint8_t *data = new uint8_t[HEX_PAGE_SIZE];
#else
    uint8_t data[HEX_PAGE_SIZE];
#endif
    uint16_t flashTail = FLASH_SIZE; // Читаем всю память до FLASH_SIZE
    uint8_t len;

    if (data) {
      for (uint16_t addr = 0; addr < flashTail; addr += HEX_PAGE_SIZE) {
        for (uint8_t i = 0; i < HEX_PAGE_SIZE; ++i) {
          data[i] = ispReadFlash(addr + i);
        }
        len = dataLength(data, HEX_PAGE_SIZE);
        if (len) {
          uint8_t crc;

          crc = len + (addr / 256) + (addr & 0xFF);
          for (uint8_t i = 0; i < len; ++i) {
            crc += data[i];
          }
          f.print(':');
          f.print(hex(name, len));
          f.print(hex(name, addr / 256));
          f.print(hex(name, addr));
          f.print(F("00"));
          for (uint8_t i = 0; i < len; ++i) {
            f.print(hex(name, data[i]));
          }
          f.println(hex(name, 0 - crc));
        }
        if (!(addr % 1024))
          printPercent((uint32_t)addr * 100 / flashTail);
      }
#ifdef USE_HEAP
      delete[] data;
#endif
      f.println(F(":00000001FF"));
      result = true;
      Serial.println(FPSTR(FAIL_OR_OK[1]));
    }
    f.close();
  }
  return result;
}



// Функция программирования Flash из файла
static bool programFlash(PGM_P fileName) {
  static const char FLASH_WRITE_ERROR[] PROGMEM = "\r\nFlash write error!";

  char name[13];
  bool result = false;

  strcpy_P(name, fileName);
  f = SPIFFS.open(name, FILE_READ);
  if (f) {
#ifdef USE_HEAP
    uint8_t *page = new uint8_t[FLASH_PAGE_SIZE * 2];
    uint8_t *data = new uint8_t[HEX_PAGE_SIZE];
#else
    uint8_t page[FLASH_PAGE_SIZE * 2], data[HEX_PAGE_SIZE];
#endif
    uint16_t pageAddr, addr;
    hexparse_t parse;
    hextype_t type;
    uint8_t len;
    bool ok;

    if (page && data) {
      pageAddr = 0xFFFF;
      ispChipErase();
      do {
        parse = parseHexLine(len, addr, type, data);
        if ((ok = (parse == HEX_OK))) {
          if (type == HEX_EXTADDR) {
            if ((len == 2) && (addr == 0)) {
              ispCommand(0x4D, 0x00, data[1], 0x00);
            } else {
              Serial.print(FPSTR(HEX_LINE_HAS));
              Serial.println(F("wrong EXTADDR!"));
              ok = false;
            }
          } else if (type == HEX_BIN) {
            if (addr + len <= FLASH_SIZE) {
              if ((addr & 0xFF80) != pageAddr) {
                if (pageAddr != 0xFFFF) {
                  if (!ispWriteFlashPage(pageAddr, page, true)) {
                    Serial.println(FPSTR(FLASH_WRITE_ERROR));
                    ok = false;
                    break;
                  }
                }
                pageAddr = addr & 0xFF80;
                memset(page, 0xFF, FLASH_PAGE_SIZE * 2);
              }
              for (uint8_t i = 0; i < len; ++i) {
                page[(addr + i) & 0x7F] = data[i];
              }
            } else {
              Serial.print(FPSTR(HEX_LINE_HAS));
              Serial.println(F("wrong flash address!"));
              ok = false;
            }
          } else if (type == HEX_END) {
            if ((len == 0) && (addr == 0)) {
              if (pageAddr != 0xFFFF) {
                ok = ispWriteFlashPage(pageAddr, page, true);
                if (!ok)
                  Serial.println(FPSTR(FLASH_WRITE_ERROR));
              }
              break;
            } else {
              Serial.print(FPSTR(HEX_LINE_HAS));
              Serial.println(F("wrong END!"));
              ok = false;
            }
          } else {
            Serial.print(FPSTR(HEX_LINE_HAS));
            Serial.println(F("unexpected type!"));
            ok = false;
          }

        } else {
          printParseError(parse);
        }

      } while (ok);
#ifdef USE_HEAP
      delete[] data;
      delete[] page;
#endif

      result = ok;
    }
    f.close();
  }
  return result;
}

// Настройка
void setup() {
  Serial.begin(115200);

  // Инициализация SPIFFS
  if (!SPIFFS.begin(true)) {
    Serial.println(F("Ошибка инициализации SPIFFS!"));
    error = true;
    return;
  }

  // Настройка пинов
  pinMode(BTN1_PIN, INPUT_PULLUP);
  pinMode(BTN2_PIN, INPUT_PULLUP);

  // Инициализация ISP
  ispInit();
}

int prog = 0;

// Основной цикл
void loop() {
  pinMode(13, OUTPUT);
  digitalWrite(13, 1); // отпускание ресета

  if (digitalRead(BTN2_PIN) == 0) {
    prog = 0;
    pinMode(2, OUTPUT);
     digitalWrite(2, 1);
  }else{
    prog = 1;
    pinMode(2, OUTPUT);
     digitalWrite(2, 1);
  }

 

  // Ожидание нажатия кнопки для начала работы
  if (digitalRead(BTN1_PIN) == 0) {
    delay(10);
digitalWrite(2, 0);
    // Основной цикл обработки нажатий кнопок
    if (ispBegin()) {
      uint8_t sign[3];
      ispReadSignature(sign);
      Serial.print(F("Cигнатура: "));
      for (uint8_t i = 0; i < 3; ++i) {
        if (i)
          Serial.print(F(", "));
        if (sign[i] < 0x10)
          Serial.print('0');
        Serial.print(sign[i], HEX);
      }
      Serial.println();
      if ((sign[0] == 0x1E) && (sign[1] == 0x95) && ((sign[2] == 0x0F) || (sign[2] == 0x14))) {
        // Обработка нажатий кнопок
        if (prog == 0) {
          Serial.println(F("Запись fuses.txt.."));
          if (programFuses(FUSES_NAME)) {
            Serial.println(F("Успешно!"));
            digitalWrite(2, 0);
            prog = 2;
            // reser(); // перезагрузка
          } else {
            Serial.println(F("Ошибка!"));
            error = true;
            reser(); // перезагрузка
          }
        }
        if (prog == 1) {
          Serial.println(F("Чтение fuses.bak.."));
          if (dumpFuses(FUSES_BACKUP_NAME)) {
            Serial.println(F("Успешно!"));
             
            prog = 3;
              // reser(); // перезагрузка
          } else {
            Serial.println(F("Ошибка!"));
            error = true;
            reser(); // перезагрузка
          }
        }
        if (prog == 2) {
          Serial.println(F("Запись eeprom.hex.."));
          if (programEeprom(EEPROM_NAME)) {
            Serial.println(F("Успешно!"));
             
            prog = 4;
            
            //reser(); // перезагрузка
          } else {
            Serial.println(F("Ошибка!"));
            error = true;
            reser(); // перезагрузка
          }
        }
        if (prog == 3) {
          Serial.println(F("Чтение eeprom.bak.."));
          if (dumpEeprom(EEPROM_BACKUP_NAME)) {
            Serial.println(F("Успешно!"));
            
            prog = 5;
            // reser(); // перезагрузка
          } else {
            Serial.println(F("Ошибка!"));
            error = true;
            reser(); // перезагрузка
          }
        }
        if (prog == 4) {
          Serial.println(F("Запись firmware.hex.."));
          if (programFlash(FIRMWARE_NAME)) {
            Serial.println(F("Успешно!"));
            
            reser(); // перезагрузка
          } else {
            Serial.println(F("Ошибка!"));
            error = true;
            reser(); // перезагрузка
          }
        }
        if (prog == 5) {
          Serial.println(F("Чтение firmware.bak.."));
          if (dumpFlash(FIRMWARE_BACKUP_NAME)) {
            Serial.println(F("Успешно!"));
            
            reser(); // перезагрузка
          } else {
            Serial.println(F("Ошибка!"));
            error = true;
            reser(); // перезагрузка
          }
        }
      } else {
        Serial.println(F("Ошибка сигнатуры!"));
        error = true;
        reser(); // перезагрузка
      }
    } else {
      Serial.println(F("MK не подключён!"));
      error = true;
      reser(); // перезагрузка
    }
    ispDone();
  }
}

void reser() {
  Serial.println("Автор - Петров Виталий Николаевич");
  Serial.println("Сайт - ardublock.ru");
  delay(2000);
  ESP.restart(); // Перезагрузка ESP32
}