#pragma once

#include <Arduino.h>

// Определение пинов для ESP32
#define ISP_RST   13 // 4  // Пример пина для RST
#define ISP_DO    27 // 5  // Пример пина для DO
#define ISP_DI    14 // 18 // Пример пина для DI
#define ISP_SCK   12 // 19 // Пример пина для SCK

constexpr uint16_t EEPROM_SIZE = 1024;
constexpr uint8_t EEPROM_PAGE_SIZE = 4;
constexpr uint16_t FLASH_SIZE = 32768;
constexpr uint8_t FLASH_PAGE_SIZE = 64;

static void ispInit() {
  pinMode(ISP_RST, OUTPUT);
  pinMode(ISP_DO, OUTPUT);
  pinMode(ISP_SCK, OUTPUT);
  pinMode(ISP_DI, INPUT);
  digitalWrite(ISP_RST, LOW);
  digitalWrite(ISP_SCK, LOW);
  digitalWrite(ISP_DI, LOW);
}

static void ispDone() {
  digitalWrite(ISP_RST, LOW);
  digitalWrite(ISP_DO, LOW);
  digitalWrite(ISP_SCK, LOW);
  pinMode(ISP_RST, INPUT);
  pinMode(ISP_DI, INPUT);
  pinMode(ISP_DO, INPUT);
  pinMode(ISP_SCK, INPUT);
}

static void ispReset() {
  digitalWrite(ISP_RST, HIGH);
  delay(1);
  digitalWrite(ISP_RST, LOW);
}

static uint8_t ispTransfer(uint8_t data) {
  uint8_t result = 0;

  for (uint8_t bit = 0; bit < 8; ++bit) {
    digitalWrite(ISP_DO, (data & 0x80) ? HIGH : LOW);
    digitalWrite(ISP_SCK, HIGH);
    delayMicroseconds(1);
    result <<= 1;
    if (digitalRead(ISP_DI) == HIGH)
      result |= 1;
    digitalWrite(ISP_SCK, LOW);
    delayMicroseconds(1);
    data <<= 1;
  }
  return result;
}

static bool ispBegin() {
  bool result;
  uint8_t retry = 5;

  while (retry--) {
    digitalWrite(ISP_RST, LOW);
    delay(20);
    ispTransfer(0xAC);
    ispTransfer(0x53);
    result = ispTransfer(0x00) == 0x53;
    ispTransfer(0x00);
    if (result)
      break;
    ispReset();
  }
  return result;
}

static uint8_t ispCommand(uint8_t cmd1, uint8_t cmd2, uint8_t cmd3, uint8_t cmd4 = 0x00) {
  ispTransfer(cmd1);
  ispTransfer(cmd2);
  ispTransfer(cmd3);
  return ispTransfer(cmd4);
}

static void ispWait() {
  while (ispCommand(0xF0, 0x00, 0x00) & 0x01) {
    delay(1);
  }
}

static inline uint8_t ispReadLockBits() {
  return ispCommand(0x58, 0x00, 0x00);
}

static inline void ispWriteLockBits(uint8_t bits) {
  ispCommand(0xAC, 0xE0, 0x00, bits);
}

static void ispReadSignature(uint8_t *sign) {
  for (uint8_t i = 0; i < 3; ++i) {
    sign[i] = ispCommand(0x30, 0x00, i);
  }
}

static inline uint8_t ispReadLowFuseBits() {
  return ispCommand(0x50, 0x00, 0x00);
}

static inline void ispWriteLowFuseBits(uint8_t bits) {
  ispCommand(0xAC, 0xA0, 0x00, bits);
}

static inline uint8_t ispReadHighFuseBits() {
  return ispCommand(0x58, 0x08, 0x00);
}

static inline void ispWriteHighFuseBits(uint8_t bits) {
  ispCommand(0xAC, 0xA8, 0x00, bits);
}

static inline uint8_t ispReadExtFuseBits() {
  return ispCommand(0x50, 0x08, 0x00);
}

static inline void ispWriteExtFuseBits(uint8_t bits) {
  ispCommand(0xAC, 0xA4, 0x00, bits);
}

static void ispChipErase() {
  ispCommand(0xAC, 0x80, 0x00, 0x00);
  ispWait();
}

static inline uint8_t ispReadEeprom(uint16_t addr) {
  return ispCommand(0xA0, addr / 256, addr);
}

static bool ispWriteEeprom(uint16_t addr, uint8_t data, bool verify = false) {
  ispCommand(0xC0, addr / 256, addr, data);
  ispWait();
  if (verify) {
    return ispReadEeprom(addr) == data;
  }
  return true;
}

static bool ispWriteEepromPage(uint16_t addr, const uint8_t *page, bool verify = false) {
  addr &= 0xFFFC;
  for (uint8_t i = 0; i < EEPROM_PAGE_SIZE; ++i) {
    ispCommand(0xC1, 0x00, i, page[i]);
  }
  ispCommand(0xC2, addr / 256, addr, 0);
  ispWait();
  if (verify) {
    for (uint8_t i = 0; i < EEPROM_PAGE_SIZE; ++i) {
      if (ispReadEeprom(addr + i) != page[i])
        return false;
    }
  }
  return true;
}

static bool ispWriteEepromPage_P(uint16_t addr, const uint8_t *page, bool verify = false) {
  addr &= 0xFFFC;
  for (uint8_t i = 0; i < EEPROM_PAGE_SIZE; ++i) {
    ispCommand(0xC1, 0x00, i, pgm_read_byte(&page[i]));
  }
  ispCommand(0xC2, addr / 256, addr, 0);
  ispWait();
  if (verify) {
    for (uint8_t i = 0; i < EEPROM_PAGE_SIZE; ++i) {
      if (ispReadEeprom(addr + i) != pgm_read_byte(&page[i]))
        return false;
    }
  }
  return true;
}

static inline uint8_t ispReadFlash(uint16_t addr) {
  return ispCommand(0x20 + 0x08 * (addr & 0x01), addr / 512, addr / 2);
}

static void ispFillFlashPage(uint16_t addr, uint16_t data = 0xFFFF) {
  addr /= 2;
  addr &= 0xFFC0;
  for (uint8_t i = 0; i < FLASH_PAGE_SIZE; ++i) {
    ispCommand(0x40, 0x00, i, data);
    ispCommand(0x48, 0x00, i, data / 256);
  }
  ispCommand(0x4C, addr / 256, addr, 0);
  ispWait();
}

static bool ispWriteFlashPage(uint16_t addr, const uint8_t *page, bool verify = false) {
  addr /= 2;
  addr &= 0xFFC0;
  for (uint8_t i = 0; i < FLASH_PAGE_SIZE; ++i) {
    ispCommand(0x40, 0x00, i, page[i * 2]);
    ispCommand(0x48, 0x00, i, page[i * 2 + 1]);
  }
  ispCommand(0x4C, addr / 256, addr, 0);
  ispWait();
  if (verify) {
    for (uint8_t i = 0; i < FLASH_PAGE_SIZE; ++i) {
      if ((ispReadFlash(addr * 2 + i * 2) != page[i * 2]) || (ispReadFlash(addr * 2 + i * 2 + 1) != page[i * 2 + 1]))
        return false;
    }
  }
  return true;
}

static bool ispWriteFlashPage_P(uint16_t addr, const uint8_t *page, bool verify = false) {
  addr /= 2;
  addr &= 0xFFC0;
  for (uint8_t i = 0; i < FLASH_PAGE_SIZE; ++i) {
    ispCommand(0x40, 0x00, i, pgm_read_byte(&page[i * 2]));
    ispCommand(0x48, 0x00, i, pgm_read_byte(&page[i * 2 + 1]));
  }
  ispCommand(0x4C, addr / 256, addr, 0);
  ispWait();
  if (verify) {
    for (uint8_t i = 0; i < FLASH_PAGE_SIZE; ++i) {
      if ((ispReadFlash(addr * 2 + i * 2) != pgm_read_byte(&page[i * 2])) ||
          (ispReadFlash(addr * 2 + i * 2 + 1) != pgm_read_byte(&page[i * 2 + 1]))) {
        return false;
      }
    }
  }
  return true;
}