#include <SoftwareSerial.h>
#include <Wire.h>
#include "ASOLED.h"

SoftwareSerial mySerial(2, 3); // RX, TX

// Конфигурация пинов
const byte BUTTON_BAUD_PIN = 12;
const byte BUTTON_TYPE_PIN = 11;
const unsigned long DEBOUNCE_DELAY = 50;
const unsigned long LINE_TIMEOUT = 100; // Таймаут конца строки (мс)
const unsigned long STARTUP_DELAY = 1000; // Задержка при старте (мс)

// Настройки скорости
const unsigned long BAUD_RATES[] = {300, 1200, 2400, 4800, 9600, 14400, 19200, 38400, 57600, 115200};
const byte BAUD_COUNT = sizeof(BAUD_RATES)/sizeof(BAUD_RATES[0]);
byte currentBaudIndex = 4; // Стартовая скорость 9600

// Типы данных
enum DataType {
  TYPE_TEXT_LINE,  // Текстовая строка (по таймауту или \n)
  TYPE_TEXT_CHAR,  // По одному символу
  TYPE_DECIMAL,    // Числовое значение 0-255
  TYPE_HEX,        // HEX представление
  TYPE_BINARY      // Бинарное представление
};
const byte TYPE_COUNT = 5;
const char* TYPE_NAMES[] = {
  "Text", "Char", "Dec", "Hex", "Bin"
};
DataType currentDataType = TYPE_TEXT_LINE;

// Буфер данных
String currentLine = ""; // Текущая собираемая строка
unsigned long lastCharTime = 0; // Время последнего символа
String displayLines[8];  // Буфер дисплея (8 строк)

// Структура для кнопки
struct Button {
  byte pin;
  byte state;
  byte lastState;
  unsigned long lastDebounceTime;
  
  bool pressed() {
    byte reading = digitalRead(pin);
    
    if (reading != lastState) {
      lastDebounceTime = millis();
    }
    
    if ((millis() - lastDebounceTime) > DEBOUNCE_DELAY) {
      if (reading != state) {
        state = reading;
        return state == LOW;
      }
    }
    
    lastState = reading;
    return false;
  }
};

Button baudButton = {BUTTON_BAUD_PIN, HIGH, HIGH, 0};
Button typeButton = {BUTTON_TYPE_PIN, HIGH, HIGH, 0};

void setup() {
  pinMode(baudButton.pin, INPUT_PULLUP);
  pinMode(typeButton.pin, INPUT_PULLUP);
  
  LD.init();
  LD.clearDisplay();
  
  // Отображение текста
  LD.printString_6x8("UART Анализатор ", 10, 2); 
  LD.printString_6x8("2 pin -> TX", 10, 4); 
  delay(STARTUP_DELAY); // Задержка 1 секунда
  
  mySerial.begin(BAUD_RATES[currentBaudIndex]);
  
  for(byte i = 0; i < 8; i++) {
    displayLines[i] = "";
  }
}

String formatData(byte data) {
  char buffer[10];
  
  switch(currentDataType) {
    case TYPE_TEXT_LINE:
    case TYPE_TEXT_CHAR:
      return String((char)data);
      
    case TYPE_DECIMAL:
      return String(data);
      
    case TYPE_HEX:
      sprintf(buffer, "0x%02X", data);
      return String(buffer);
      
    case TYPE_BINARY: {
      String res;
      for(byte i = 0; i < 8; i++) {
        res += (data & (1 << (7-i))) ? '1' : '0';
      }
      return res;
    }
    
    default:
      return "?";
  }
}

void scrollBuffer(String newLine) {
  for(byte i = 7; i > 0; i--) {
    displayLines[i] = displayLines[i-1];
  }
  displayLines[0] = newLine;
}

void updateDisplay(bool flash = false) {
  LD.clearDisplay();
  
  // Заголовок с настройками
  String header = String(BAUD_RATES[currentBaudIndex]);
  String header1 = String(TYPE_NAMES[currentDataType]);
  LD.printString_6x8(header.c_str(), 90, 0);
  LD.printString_6x8(header1.c_str(), 90, 1);
  
  // Данные
  for(byte i = 0; i < 7; i++) {
    if(displayLines[i].length() > 0) {
      LD.printString_6x8(displayLines[i].c_str(), 0, 6-i);
    }
  }
  
  if(flash) {
    delay(300); // Визуальное подтверждение изменения
  }
}

void processCompleteLine() {
  if(currentLine.length() > 0) {
    if(currentLine.length() > 30) {
      currentLine = currentLine.substring(0, 30) + "...";
    }
    scrollBuffer(currentLine);
    currentLine = "";
    updateDisplay();
  }
}

void loop() {
  // Обработка кнопки скорости
  if(baudButton.pressed()) {
    currentBaudIndex = (currentBaudIndex + 1) % BAUD_COUNT;
    mySerial.begin(BAUD_RATES[currentBaudIndex]);
    updateDisplay(true);
  }
  
  // Обработка кнопки типа данных
  if(typeButton.pressed()) {
    currentDataType = (DataType)((currentDataType + 1) % TYPE_COUNT);
    updateDisplay(true);
  }
  
  // Прием данных
  if(mySerial.available()) {
    byte data = mySerial.read();
    lastCharTime = millis();
    
    if(currentDataType == TYPE_TEXT_LINE) {
      if(data == '\n' || data == '\r') {
        processCompleteLine();
      } else if(data >= 32 && data <= 126) { // Только печатные символы
        currentLine += (char)data;
      }
    } else {
      String formatted = formatData(data);
      scrollBuffer(formatted);
      updateDisplay();
    }
  }
  
  // Проверка таймаута для текстовой строки
  if(currentDataType == TYPE_TEXT_LINE && currentLine.length() > 0) {
    if(millis() - lastCharTime > LINE_TIMEOUT) {
      processCompleteLine();
    }
  }
}
